<?php
// inc/functions.php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/config.php';

// ---------------------------------------------------------------------
// SECURE SESSION START
// ---------------------------------------------------------------------
if (session_status() === PHP_SESSION_NONE) {
    session_start([
        'cookie_httponly' => true,
        'cookie_secure'   => isset($_SERVER['HTTPS']),
        'cookie_samesite' => 'Strict'
    ]);
}

// ---------------------------------------------------------------------
// BASIC ESCAPE OUTPUT
// ---------------------------------------------------------------------
function e($str) { 
    return htmlspecialchars($str, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); 
}

// ---------------------------------------------------------------------
// SAFE REDIRECT
// ---------------------------------------------------------------------
function redirect($url) {
    header("Location: $url");
    exit();
}

// ---------------------------------------------------------------------
// CSRF PROTECTION
// ---------------------------------------------------------------------
function generate_csrf_token() {
    $token = bin2hex(random_bytes(32));
    $_SESSION['csrf'] = $token;
    return $token;
}

function verify_csrf_token($token) {
    return isset($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], $token);
}

// ---------------------------------------------------------------------
// AUTH HELPERS
// ---------------------------------------------------------------------
function admin_logged_in() {
    return !empty($_SESSION['admin_id']);
}

function require_admin() {
    if (!admin_logged_in()) redirect("login.php");
}

// ---------------------------------------------------------------------
// EMAIL VALIDATION
// ---------------------------------------------------------------------
function is_valid_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// ---------------------------------------------------------------------
// SAFE INPUT SANITIZATION
// ---------------------------------------------------------------------
function sanitize_string($str) {
    return trim(filter_var($str, FILTER_SANITIZE_STRING));
}

function sanitize_int($num) {
    return filter_var($num, FILTER_VALIDATE_INT);
}

function sanitize_float($num) {
    return filter_var($num, FILTER_VALIDATE_FLOAT);
}

// ---------------------------------------------------------------------
// PAGINATION HELPER
// ---------------------------------------------------------------------
function paginate($page, $limit = 12) {
    $page = max(1, intval($page));
    $offset = ($page - 1) * $limit;
    return [$offset, $limit];
}

// ---------------------------------------------------------------------
// LOGGING
// ---------------------------------------------------------------------
function system_log($message) {
    $logFile = __DIR__ . "/system.log";
    error_log(date("Y-m-d H:i:s") . " - $message\n", 3, $logFile);
}

// ---------------------------------------------------------------------
// STRONG IMAGE VALIDATION
// ---------------------------------------------------------------------
function validate_image_file($tmp, $name) {
    $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));

    // BLOCK dangerous extensions
    $disallowed = ['php','js','html','exe','sh','py'];
    if (in_array($ext, $disallowed)) return false;

    // Extension allowed?
    if (!in_array($ext, ALLOWED_EXT)) return false;

    // Size check
    if (filesize($tmp) > MAX_IMAGE_SIZE) return false;

    // MIME check (real MIME)
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = finfo_file($finfo, $tmp);
    finfo_close($finfo);

    $allowed_mimes = ['image/jpeg','image/png','image/webp'];
    if (!in_array($mime, $allowed_mimes)) return false;

    // getimagesize check
    if (!getimagesize($tmp)) return false;

    return true;
}

// ---------------------------------------------------------------------
// UPLOAD MULTIPLE IMAGES (SECURE)
// ---------------------------------------------------------------------
function validate_and_upload_images($files, $car_id = null) {
    if (!is_dir(UPLOAD_DIR)) mkdir(UPLOAD_DIR, 0755, true);

    $uploaded = [];
    $count = count($files['name']);

    for ($i = 0; $i < $count; $i++) {
        if ($files['error'][$i] !== UPLOAD_ERR_OK) continue;

        $tmp  = $files['tmp_name'][$i];
        $name = basename($files['name'][$i]);

        if (!validate_image_file($tmp, $name)) continue;

        // Unique filename
        $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
        $newName = time() . '_' . bin2hex(random_bytes(10)) . '.' . $ext;
        $dest = UPLOAD_DIR . '/' . $newName;

        if (move_uploaded_file($tmp, $dest)) {
            $uploaded[] = $newName;

            // Optional: if car_id provided, insert into DB
            if ($car_id !== null) {
                global $pdo;
                $stmt = $pdo->prepare("INSERT INTO car_images (car_id, image) VALUES (?, ?)");
                $stmt->execute([$car_id, $newName]);
            }
        }
    }

    return $uploaded;
}

// ---------------------------------------------------------------------
// SAFE DATABASE FETCH HELPERS
// ---------------------------------------------------------------------
function fetch_car($car_id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM cars WHERE id = ? LIMIT 1");
    $stmt->execute([$car_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function fetch_car_images($car_id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM car_images WHERE car_id = ?");
    $stmt->execute([$car_id]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// ---------------------------------------------------------------------
// SEARCH + FILTER HELPER
// ---------------------------------------------------------------------
function build_search_query($keyword, $brand, $model, $min_price, $max_price, $min_year, $max_year, $seller_type) {
    $where = [];
    $params = [];

    if (!empty($keyword)) {
        $where[] = "(brand LIKE ? OR model LIKE ? OR description LIKE ?)";
        $params[] = "%$keyword%";
        $params[] = "%$keyword%";
        $params[] = "%$keyword%";
    }
    if ($brand) {
        $where[] = "brand = ?";
        $params[] = $brand;
    }
    if ($model) {
        $where[] = "model = ?";
        $params[] = $model;
    }
    if ($seller_type) {
        $where[] = "seller_type = ?";
        $params[] = $seller_type;
    }
    if ($min_price) {
        $where[] = "price >= ?";
        $params[] = $min_price;
    }
    if ($max_price) {
        $where[] = "price <= ?";
        $params[] = $max_price;
    }
    if ($min_year) {
        $where[] = "year >= ?";
        $params[] = $min_year;
    }
    if ($max_year) {
        $where[] = "year <= ?";
        $params[] = $max_year;
    }

    $sql = "SELECT * FROM cars";
    if ($where) $sql .= " WHERE " . implode(" AND ", $where);
    $sql .= " ORDER BY id DESC";

    return [$sql, $params];
}
